////////////////////////
// sequence_engine.js //
//////////////////////////////////
// Dependencies: navnode.js     //
////////////////////////////////////////////////////////////////////////
// The NavSeqEngine object is an engine designed
// for the sole purpose of determining the next/previous
// node in a lesson tree.  The algorithm used here for determining
// the next/previous node is the PARENT_SIBLING algorithm.
// The PARENT_SIBLING algorithm works in such a way as to
// mimic the flow of UNITECH's standard courseware.
//
// Member Properties:
//
// blockedNode
//
// Member Functions:
//
// firstNodeInSequence()
// nextInSequence()
// previousInSequence()

function NavSeqEngine(navEngine)
{
	this.navEngine = navEngine;
	this.firstNodeInSequence = SequenceEngineFirstNodeInSequence;
	this.nextInSequence = SequenceEngineNextInSequence;
	this.previousInSequence = SequenceEnginePreviousInSequence;
	
	this.next = Parent_Sibling_Next;
	this.previous = Parent_Sibling_Previous;
	this.nextDown = First_Left_Down;
	this.previousDown = First_Right_Down;
}

function SequenceEngineFirstNodeInSequence(nodeRef)
{
	if (nodeRef == null) return null;
	
	this.navEngine.blockedNode = null;
	var retVal = this.nextDown(nodeRef);
	if (retVal == null)
	{
		retVal = this.next(nodeRef,false);
	}
	return retVal;
}

function SequenceEngineNextInSequence(nodeRef, bolAdjust)
{
	if (nodeRef == null) return null;
	if (bolAdjust == null) bolAdjust = false;
	
	this.navEngine.blockedNode = null;
	return this.next(nodeRef, bolAdjust);
}

function SequenceEnginePreviousInSequence(nodeRef, bolAdjust)
{
	if (nodeRef == null) return null;
	if (bolAdjust == null) bolAdjust = false;
	
	this.navEngine.blockedNode = null;
	return this.previous(nodeRef, bolAdjust);
}

function Parent_Sibling_Next(ptrCurrent, bolAdjust)
{
	var retVal = new Object();
	retVal.destination = null;
	retVal.returnLocation = null;

	var postAction = ptrCurrent.getPostAction(NAV_DIRECTION_FORWARD);
	if ((postAction.value & NAV_POST_ACTION_RESET) == NAV_POST_ACTION_RESET)
	{
		if (bolAdjust)
		{
			ptrCurrent.reset();
		}
		postAction.value -= NAV_POST_ACTION_RESET;
	}
	
	switch (postAction.value)
	{
		case NAV_POST_ACTION_RETRY:retVal.destination=ptrCurrent;return retVal;break;
		case NAV_POST_ACTION_NEXT:break;
		case NAV_POST_ACTION_PREVIOUS:return this.previous(ptrCurrent,bolAdjust);break;
		case NAV_POST_ACTION_GOTO:
			if (this.navEngine.evalGotoNode(postAction.parameters[0])==NAV_DENY_NONE)
			{
				retVal.destination = this.firstNodeInSequence(this.navEngine.getNodeById(postAction.parameters[0]));
				if (retVal.destination == null)
				{
					return null;
				}
				else
				{
					return retVal;
				}
			}
			else
			{
				return null;
			}
			break;
		case NAV_POST_ACTION_GOSUB:retVal.destination=this.navEngine.getNodeById(postAction.parameters[0]);retVal.returnLocation=ptrCurrent;return retVal;break;
		case NAV_POST_ACTION_GOSUB_RETURN:retVal.destination=this.navEngine.getNodeById(postAction.parameters[0]);retVal.returnLocation=this.navEngine.getNodeById(postAction.parameters[1]);return retVal;break;
	}
	
	// If you've reached the root node, then you
	// cannot go any further up in the Parent-Sibling
	// progression (since there is no parent) and
	// hence no siblings, so null is returned.
	if (ptrCurrent.parentStack.length == 0)
	{
		return null;
	}
	
	if (ptrCurrent.getBlock())
	{
		this.navEngine.blockedNode = ptrCurrent;
		return null;
	}
	
	var ptrParent = ptrCurrent.parentStack[ptrCurrent.parentStack.length-1];
	if (bolAdjust && ((this.navEngine.isRootNode(ptrCurrent) && ptrCurrent.parentStack.length > 0) || (!this.navEngine.isRootNode(ptrCurrent) && ptrCurrent.parentStack.length > 1)))
	{
		ptrCurrent.parentStack.length--;
	}
	
	// If the parent has a non-null url, then
	// it is the next node in the sequence,
	// and a reference to it is returned.
	if (ptrParent.url != "")
	{
		retVal.destination = ptrParent;
		return retVal;
	}
	
	// If execution reaches this point, then:
	//  the parent is non-null
	//  the parent has no url.
	
	var intStartIndex = null;
	var parentChildArray;
	if (ptrParent.selectedChildren.length > 0)
	{
		parentChildArray = ptrParent.selectedChildren;
	}
	else
	{
		parentChildArray = ptrParent.children;
	}
	// Find the index of ptrCurrent in ptrParent.children,
	// and add 1 to it to determine which siblings
	// to process (since we only want to check those
	// siblings which are to the right).
	for (var i=0;i<parentChildArray.length-1;i++)
	{
		if (parentChildArray[i] == ptrCurrent)
		{
			intStartIndex = i+1;
			break;
		}
	}
	
	var ptrNextNode = null;
	// If intStartIndex == null, then there are no siblings to process.
	if (intStartIndex != null)
	{
		// Traverse through siblings one at a time
		for (var i=intStartIndex;i<parentChildArray.length;i++)
		{
			if ( parentChildArray[i].getEnabled() && !parentChildArray[i].getSkip() )
			{
				// Check to see if the sibling is, or contains, a node
				// with a non-null url property.
				ptrNextNode = this.nextDown(parentChildArray[i])
				if (this.navEngine.blockedNode != null)
				{
					return null;
				}
				if (ptrNextNode != null)
				{
					break;
				}
			}
			
			if (parentChildArray[i].getBlock())
			{
				this.navEngine.blockedNode = parentChildArray[i];
				return null;
			}
		}
	}
	
	// ptrNextNode == null if there are no siblings that are,
	// or contain, an enabled, non-skip node with a non-null url property.
	// If no valid node was found in any of the siblings, then
	// recurse on the parent node.
	if (ptrNextNode == null)
	{
		return this.next(ptrParent,bolAdjust);	
	}
	
	retVal.destination = ptrNextNode;
	return retVal;
}

function First_Left_Down(ptrNode)
{
	// If ptrNode has a non-null url, then
	// it is the next node in the sequence,
	// and a reference to it is returned.
	if (ptrNode.url != "")
	{
		return ptrNode;
	}
	
	var ptrNextNode = null;
	var childArray;
	if (ptrNode.selectedChildren.length > 0)
	{
		childArray = ptrNode.selectedChildren;
	}
	else
	{
		childArray = ptrNode.children;
	}
	for (var i=0;i<childArray.length;i++)
	{
		if (childArray[i].getEnabled() && !childArray[i].getSkip())
		{
			ptrNextNode = this.nextDown(childArray[i]);
			if (this.navEngine.blockedNode != null)
			{
				return null;
			}
			if (ptrNextNode != null)
			{
				break;
			}
		}
		
		if (childArray[i].getBlock())
		{
			this.navEngine.blockedNode = childArray[i];
			return null;
		}
	}

	return ptrNextNode;
}

function Parent_Sibling_Previous(ptrCurrent, bolAdjust)
{
	var retVal = new Object();
	retVal.destination = null;
	retVal.returnLocation = null;
	
	var postAction = ptrCurrent.getPostAction(NAV_DIRECTION_REVERSE);
	if ((postAction.value & NAV_POST_ACTION_RESET) == NAV_POST_ACTION_RESET)
	{
		if (bolAdjust)
		{
			ptrCurrent.reset();
		}
		postAction.value -= NAV_POST_ACTION_RESET;
	}
	
	switch (postAction.value)
	{
		case NAV_POST_ACTION_RETRY:retVal.destination=ptrCurrent;return retVal;break;
		case NAV_POST_ACTION_NEXT:return this.next(ptrCurrent,bolAdjust);break;
		case NAV_POST_ACTION_PREVIOUS:break;
		case NAV_POST_ACTION_GOTO:
			if (this.navEngine.evalGotoNode(postAction.parameters[0])==NAV_DENY_NONE)
			{
				retVal.destination = this.firstNodeInSequence(this.navEngine.getNodeById(postAction.parameters[0]));
				if (retVal.destination == null)
				{
					return null;
				}
				else
				{
					return retVal;
				}
			}
			else
			{
				return null;
			}
			break;
		case NAV_POST_ACTION_GOSUB:retVal.destination=this.navEngine.getNodeById(postAction.parameters[0]);retVal.returnLocation=ptrCurrent;return retVal;break;
		case NAV_POST_ACTION_GOSUB_RETURN:retVal.destination=this.navEngine.getNodeById(postAction.parameters[0]);retVal.returnLocation=this.navEngine.getNodeById(postAction.parameters[1]);return retVal;break;
	}

	// If you've reached the root node, then you
	// cannot go any further up in the Parent-Sibling
	// progression (since there is no parent) and
	// hence no siblings, so null is returned.
	if (ptrCurrent.parentStack.length == 0)
	{
		return null;
	}
	
	if (ptrCurrent.parentStack[0].cForwardOnly)
	{
		this.navEngine.blockedNode = ptrCurrent;
		return null;
	}

	var ptrParent = ptrCurrent.parentStack[ptrCurrent.parentStack.length-1];
	if (bolAdjust && ((this.navEngine.isRootNode(ptrCurrent) && ptrCurrent.parentStack.length > 0) || (!this.navEngine.isRootNode(ptrCurrent) && ptrCurrent.parentStack.length > 1)))
	{
		ptrCurrent.parentStack.length--;
	}

	// If the parent has a non-null url, then
	// it is the next node in the sequence,
	// and a reference to it is returned.
	if (ptrParent.url != "")
	{
		retVal.destination = ptrParent;
		return retVal;
	}
	
	// If execution reaches this point, then:
	//  the parent is non-null
	//  the parent has no url.
	
	var intStartIndex = null;
	var parentChildArray;
	if (ptrParent.selectedChildren.length > 0)
	{
		parentChildArray = ptrParent.selectedChildren;
	}
	else
	{
		parentChildArray = ptrParent.children;
	}
	// Find the index of ptrCurrent in ptrParent.children,
	// and subtract 1 from it to determine which siblings
	// to process (since we only want to check those
	// siblings which are to the left).
	for (var i=1;i<parentChildArray.length;i++)
	{
		if (parentChildArray[i] == ptrCurrent)
		{
			intStartIndex = i-1;
			break;
		}
	}
	
	var ptrPreviousNode = null;
	// If intStartIndex == null, then there are no siblings to process.
	if (intStartIndex != null)
	{
		// Traverse through siblings one at a time
		for (var i=intStartIndex;i>-1;i--)
		{
			if (parentChildArray[i].getEnabled() && !parentChildArray[i].getSkip())
			{
				// Check to see if the sibling is, or contains, a node
				// with a non-null url property.
				ptrPreviousNode = this.previousDown(parentChildArray[i])
				if (ptrPreviousNode != null)
				{
					break;
				}
			}
		}
	}
	
	// ptrPreviousNode == null if there are no siblings that are,
	// or contain, a node with a non-null Url property.
	// If no valid node was found in any of the siblings, then
	// recurse on the parent node.
	if (ptrPreviousNode == null)
	{
		return this.previous(ptrParent,bolAdjust);	
	}
	
	retVal.destination = ptrPreviousNode;
	return retVal;
}

function First_Right_Down(ptrNode)
{
	// If ptrNode has a non-null url, then
	// it is the next node in the sequence,
	// and a reference to it is returned.
	if (ptrNode.url != "")
	{
		return ptrNode;
	}
	
	var ptrPreviousNode = null;
	var childArray;
	if (ptrNode.selectedChildren.length > 0)
	{
		childArray = ptrNode.selectedChildren;
	}
	else
	{
		childArray = ptrNode.children;
	}
	for (var i=childArray.length-1;i>-1;i--)
	{
		if (childArray[i].getEnabled() && !childArray[i].getSkip())
		{
			ptrPreviousNode = this.previousDown(childArray[i]);
			if (ptrPreviousNode != null)
			{
				break;
			}
		}
	}

	return ptrPreviousNode;
}